/******************** (C) COPYRIGHT 2007 STMicroelectronics ********************
* File Name          : EEPROM.c
* Author             : MCD APPLICATION TEAM
* Date First Issued  : 05/30/2007 : Version 1.0
* Description        : EEPROM emulation routines
********************************************************************************
* History:
* 05/30/2007 : Version 1.0
********************************************************************************
THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME. AS A
RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR
CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT OF SUCH
SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION CONTAINED
HEREININ CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/


/* Standard include ----------------------------------------------------------*/
#include "91x_lib.h"
#include "EEPROM.h"

/* Include of other module interface headers ---------------------------------*/
/* Local includes ------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

/*Virtual address defined by the user: Note address 0xFFFF is prohibited*/

u16 Tab[max_varia] = {0xAAAA, 0x5555, 0xDDAA};

/* Private function prototypes -----------------------------------------------*/

/* Private functions ---------------------------------------------------------*/
u8 FindValidSector(u8 operation);
u16 WriteVerifyVariableFull(u16 VirtAddress, u16 Data);
u16 EepromSectorTransfer(u16 VirtAddress, u16 Data);


/*******************************************************************************
* Routine :  EepromFormat
* Purpose :  Format the whole EEPROM
* Input   :
* Output  : Status of the operation
*******************************************************************************/

u8 EepromFormat(void)
{
  u8 Status;
  u8 FMI_Timeout_Status1 ;
  u8 FMI_Timeout_Status2 ;

  /* Erase sector0 */
  FMI_EraseSector(FMI_B1S0);
  FMI_Timeout_Status1 = FMI_WaitForLastOperation(FMI_BANK_1);


  if (FMI_Timeout_Status1 == 1)
  {

   /*Declare sector0 as VALID_SECTOR:Write VALID_SECTOR at sector0 baseaddress*/
    FMI_WriteHalfWord( FMI_B1S0, 0xCCCC);
    FMI_Timeout_Status2 = FMI_WaitForLastOperation(FMI_BANK_1);


    /* If program operation was failed, a Flash error code is returned*/
    if (FMI_Timeout_Status2 != 1)
      Status = FMI_Timeout_Status2;
  }
  else
  {
    /*If erase operation is not successful so return Flash error code */
    Status = FMI_Timeout_Status1;
  }
  /* Erase sector1 */
  FMI_EraseSector(FMI_B1S1);
  FMI_Timeout_Status1 = FMI_WaitForLastOperation(FMI_BANK_1);
  Status = FMI_Timeout_Status1;
  return Status;

}

/*******************************************************************************
* Routine:  FindValidSector
* Purpose:  Find valid sector for write or read operation
* Input  :  a byte indicating whether we search valid sector for read from or
*           write into( READ_FROM_VALID_SECTOR or WRITE_IN_VALID_SECTOR
* Output:   Sector number
*******************************************************************************/

u8 FindValidSector(u8 operation)
{
  u16  STATUS0, STATUS1;
  /* Check sector 0 for status */
  STATUS0 = FMI_ReadWord(FMI_B1S0);

  /* Check sector 1 for status */
  STATUS1 = FMI_ReadWord(FMI_B1S1);
  /*write or read: either we want to write in or read from the suitable sector*/
  switch (operation)
  {
    case WRITE_IN_VALID_SECTOR:  /* write operation */
      if ( STATUS1 == VALID_SECTOR)    /* sector1 valid*/
      {
        /* sector0 receiving data */
        if ( STATUS0 == RECEIVE_DATA)  return (SECTOR0);
        else return (SECTOR1);
      }
      else if ( STATUS0 == VALID_SECTOR) /* sector0 valid */
      {
        /* sector1 receiving data */
        if ( STATUS1 == RECEIVE_DATA) return (SECTOR1);
        else return (SECTOR0);
      }
      else return No_Valid_Sector;
    case READ_FROM_VALID_SECTOR:   /* read operation */
      /* sector0 valid */
      if ( STATUS0 == VALID_SECTOR)   return (SECTOR0);
      /* sector1 valid */
      else if ( STATUS1 == VALID_SECTOR)  return (SECTOR1);
      else return No_Valid_Sector ;

    default:
      return (SECTOR0);
  }/* End switch */
} /* End  FindValidSector */

/*******************************************************************************
* Routine:  WriteVerifyVariableFull
* Purpose:  Writes  variable in EEPROM.
* Input  :
*           - VirtAddress: 16 bit virtual address
*           - Data: 16 bit data to be written
* Output:   Returns 1 on success or 0x80 if variable is full or Flash error code
*******************************************************************************/

u16 WriteVerifyVariableFull(u16 VirtAddress, u16 Data)

{

  u8 valid_sector;
  s16 status0, status, status1, status3 = -1;
  u32 address, variable_end_address;
  /* Get valid sector for write operation */
  valid_sector = FindValidSector(WRITE_IN_VALID_SECTOR);
  if (valid_sector == No_Valid_Sector) return  No_Valid_Sector;

  /* Get the variable start address in the valid sector */
  address =  (u32)(StartAddress + (u32)(valid_sector * SECTOR_SIZE));
  /* Get the variable end address in the valid sector */
  variable_end_address = (u32)((u32)(FMI_BANK_1 - 2) + SECTOR_SIZE);
  variable_end_address += (u32)(valid_sector * SECTOR_SIZE);
  do
  {
    /* verify if variable value already exists?*/

    if (((u16)(FMI_ReadWord(address + 2)) == VirtAddress)
        && ((u16)(FMI_ReadWord(address)) == Data))

      status0 = 0;
    else status0 = 0xBADC;


    /*verify if address content is 0xFFFF ; if yes status = 0,
          if no status = 0xBADC*/

    if ((u16)(FMI_ReadWord(address)) == 0xFFFF) status = 0 ;
    else status = 0xBADC;

    /* verify if (address + 2) content is 0xFFFF ; if yes status1 = 0,
             if no status1 = 0xBADC*/

    if ((u16)(FMI_ReadWord(address + 2)) == 0xFFFF)
      status1 = 0 ;
    else status1 = 0xBADC;


    if (status0 == 0)
    {
      /* VARIABLE_EXIST*/

      break;
    }


    if ((status == 0) && (status1 == 0))
    {


      FMI_WriteHalfWord( address, Data);
      status3 = FMI_WaitForLastOperation(FMI_BANK_1);

      FMI_WriteHalfWord( address + 2, VirtAddress);
      status3 = FMI_WaitForLastOperation(FMI_BANK_1);

      break;
    }

    else address = address + 4;
    if (address >= variable_end_address)
    {
      /* 0x80 is returned */
      status3 = VARIABLE_FULL;
      break;
    }
  }
  while (address < variable_end_address);  /* End do */
  return status3;

}  /* End WriteVerifyVariableFull */
/******************************************************************************
*Routine:  ReadVariable
*Purpose:  Reads the last stored variable data
*Input  : - VirtAddress:Variable virtual address
* Output:  Returns 16 bit read data on success or error code on failure
*******************************************************************************/

u16 ReadVariable(u16 VirtAddress)
{
  u8 valid_sector;
  u32 address, base_address, variable_end_address;
  u16 Read_data, Read_Address;
  /* Get active sector  for READ operation */
  valid_sector = FindValidSector(READ_FROM_VALID_SECTOR);
  if (valid_sector == No_Valid_Sector) return  No_Valid_Sector;
  /* Get variable base address in the valid sector  */
  base_address =  (u32)(StartAddress + (u32)(valid_sector * SECTOR_SIZE));
  /* Get the variable end address in the valid sector */
  variable_end_address = (u32)(FMI_BANK_1  + SECTOR_SIZE);
  variable_end_address += (u32)(valid_sector * SECTOR_SIZE);
  address = variable_end_address;

  do
  {

    Read_Address = (u16) FMI_ReadWord(address);

    if (Read_Address == VirtAddress)

    {
      Read_data =  (u16)FMI_ReadWord(address - 2);
      break;
    }
    /* End if */
    else /* Read_Address = VirtAddress */

      address = (u32)(address - 4);


  }
  while (address > (base_address + 2)); /* End do */

  return Read_data;

}/*End ReadVariable */

/*******************************************************************************
* Routine:  EepromSectorTransfer
* Purpose:  Transfers data from full sector to an empty one.
* Input  :  
*           - VirtAddress: Variable virtual address
*           - Data: 16 bit data to be written
* Output:   returns 1 on success or error code
*******************************************************************************/

u16 EepromSectorTransfer(u16 VirtAddress, u16 Data)
{
  u16 data;
  u8 valid_sector, j = 0;
  u16 status = 0;
  u32  new_sector; /* The sector we are moving to */
  u32  old_sector; /* The sector we are moving from */
  /* Get active sector  (This is the sector we are moving from) */
  valid_sector = FindValidSector(READ_FROM_VALID_SECTOR);
  if (valid_sector == SECTOR1)
  {
    new_sector = SECTOR0_BASE_ADDRESS;  /* move data to this sector*/
    old_sector = SECTOR1_BASE_ADDRESS;  /*move data from this sector*/

  }
  else if (valid_sector == SECTOR0)
  {
    new_sector = SECTOR1_BASE_ADDRESS; /* move data to this sector*/
    old_sector = SECTOR0_BASE_ADDRESS; /* move data from this sector*/

  }
  else return  No_Valid_Sector;
  /* Mark new sector to receive */

  FMI_WriteHalfWord( new_sector, RECEIVE_DATA);
  status = FMI_WaitForLastOperation(FMI_BANK_1);


  /* if error, return error code */
  if (status != 1) return status;
  /* Transfer process */
  /* Write the variable data */
  status = WriteVerifyVariableFull(VirtAddress, Data);
  if (status != 1)
    return status;
  /* Transfer other variables last updates */
  do
  {


    /* Read the other last variable updates */
    data = ReadVariable(Tab[j]);
    status = WriteVerifyVariableFull(Tab[j], data);
    if (status != 1)
      return status;


    j++;
  }
  while ((j < max_varia)&(Tab[j] != VirtAddress));
  /* Mark good and bad sectors */
  /* Mark old sector TRANSFER_COMPLETE before marking new sector VALID_SECTOR */

  FMI_WriteHalfWord( old_sector, TRANSFER_COMPLETE);
  status = FMI_WaitForLastOperation(FMI_BANK_1);

  /* Program routine returns 1 on success and Flash error code on failure.*/
  if (status != 1) return status;
  /* mark new sector VALID_SECTOR after marking old sector TRANSFER_COMPLETE*/
  FMI_WriteHalfWord( new_sector, VALID_SECTOR);
  status = FMI_WaitForLastOperation(FMI_BANK_1);


  /* Program routine returns 1 on success and Flash error code on failure.*/
  if (status != 1) return status;
  
  /* Erase the old sector */
   FMI_EraseSector(old_sector);
   
  /* Wait until erase operation is complete */
  status = FMI_WaitForLastOperation(FMI_BANK_1);


  /*If the erase operation is successful, 1 is returned.If the erase operation*/
  /*is failed, a Flash error code is returned.*/
  return status;
}
/* End EepromSectorTransfer*/


/*******************************************************************************
* Routine:  WriteVariable
* Purpose:  Writes/upadtes  variable data in EEPROM.
*Input      - VirtAddress: Variable virtual address
*           - Data: 16 bit data to be written
* Output:   returns 1 on success or error code on failure
*******************************************************************************/
u8 WriteVariable(u16 VirtAddress, u16 Data)
{
  u16 status;
  status = WriteVerifyVariableFull(VirtAddress, Data);
  /* When there isn't enough memory space for the new variable value, the new data
  variable + the last variables' updates are transferred to a blank sector.*/
  if (status == VARIABLE_FULL)
  {
    /* Perform sector transfer*/
    status = EepromSectorTransfer(VirtAddress, Data);
  }
  return status;
}/* End WriteVariable */

/******************* (c) 2007  ST Microelectronics *********** END OF FILE ****/
